import { NavigationService } from '../navigation.service';
import { GraphService } from '../graph.service';
import { PathfindService } from '../pathfind.service';
import {
  Graph,
  GraphNode,
  PathType,
  NearestReturnType,
} from '../navigation.entity';

describe('NavigationService', () => {
  let navigationService: NavigationService;
  let graphService: jest.Mocked<GraphService>;
  let pathfindService: jest.Mocked<PathfindService>;

  beforeEach(() => {
    graphService = {
      getGraph: jest.fn(),
      getNode: jest.fn(),
      getNearestNode: jest.fn(),
    } as any;

    pathfindService = {
      AStar: jest.fn(),
    } as any;

    navigationService = new NavigationService(graphService, pathfindService);
  });

  describe('getGraph', () => {
    it('should return graph from graphService', async () => {
      const mockGraph: Graph = {
        id: 'graph1',
        originX: 0,
        originY: 0,
        originZ: 0,
        scale: 1,
        nodes: [],
      };
      graphService.getGraph.mockResolvedValue(mockGraph);

      const result = await navigationService.getGraph('graph1');
      expect(graphService.getGraph).toHaveBeenCalledWith('graph1');
      expect(result).toBe(mockGraph);
    });

    it('should return null if graph not found', async () => {
      graphService.getGraph.mockResolvedValue(null);
      const result = await navigationService.getGraph('invalidId');
      expect(result).toBeNull();
    });
  });

  describe('getNode', () => {
    it('should return node from graphService', async () => {
      const mockNode: GraphNode = {
        id: 'node1',
        label: 'Node 1',
        x: 0,
        y: 0,
        z: 0,
        graphId: 'graph1',
        edges: [],
      };
      graphService.getNode.mockResolvedValue(mockNode);

      const result = await navigationService.getNode('node1');
      expect(graphService.getNode).toHaveBeenCalledWith('node1');
      expect(result).toBe(mockNode);
    });

    it('should return null if node not found', async () => {
      graphService.getNode.mockResolvedValue(null);

      const result = await navigationService.getNode('invalidNodeId');
      expect(result).toBeNull();
    });
  });

  describe('getPath', () => {
    const fromNode: GraphNode = {
      id: 'from',
      label: 'From Node',
      x: 0,
      y: 0,
      z: 0,
      graphId: 'graph1',
      edges: [],
    };
    const toNode: GraphNode = {
      id: 'to',
      label: 'To Node',
      x: 1,
      y: 1,
      z: 1,
      graphId: 'graph1',
      edges: [],
    };
    const graph: Graph = {
      id: 'graph1',
      originX: 0,
      originY: 0,
      originZ: 0,
      scale: 1,
      nodes: [fromNode, toNode],
    };
    const path: GraphNode[] = [fromNode, toNode];

    beforeEach(() => {
      graphService.getNode.mockImplementation(async (id: string) => {
        if (id === fromNode.id) return fromNode;
        if (id === toNode.id) return toNode;
        return null;
      });
      graphService.getGraph.mockResolvedValue(graph);
      pathfindService.AStar.mockReturnValue(path);
    });

    it('should throw if fromNode or toNode is not found', async () => {
      graphService.getNode.mockResolvedValueOnce(null); // fromNode not found
      await expect(
        navigationService.getPath('from', 'to', PathType.DEEP),
      ).rejects.toThrow('Invalid nodes for pathfinding');

      graphService.getNode
        .mockResolvedValueOnce(fromNode)
        .mockResolvedValueOnce(null); // toNode not found
      await expect(
        navigationService.getPath('from', 'to', PathType.DEEP),
      ).rejects.toThrow('Invalid nodes for pathfinding');
    });

    it('should throw if nodes belong to different graphs', async () => {
      graphService.getNode.mockResolvedValueOnce({
        id: 'from',
        label: 'From Node',
        x: 0,
        y: 0,
        z: 0,
        graphId: 'graph1',
        edges: [],
      });
      graphService.getNode.mockResolvedValueOnce({
        id: 'to',
        label: 'To Node',
        x: 1,
        y: 1,
        z: 1,
        graphId: 'graph2',
        edges: [],
      });
      await expect(
        navigationService.getPath('from', 'to', PathType.DEEP),
      ).rejects.toThrow('Nodes must belong to the same graph for pathfinding');
    });

    it('should throw if graph not found', async () => {
      graphService.getGraph.mockResolvedValueOnce(null);
      await expect(
        navigationService.getPath('from', 'to', PathType.DEEP),
      ).rejects.toThrow('Graph not found for pathfinding');
    });

    it('should return full nodes path for DEEP path type', async () => {
      const result = await navigationService.getPath(
        'from',
        'to',
        PathType.DEEP,
      );
      expect(result).toEqual(path);
    });

    it('should return only node IDs for SHALLOW path type', async () => {
      const result = await navigationService.getPath(
        'from',
        'to',
        PathType.SHALLOW,
      );
      expect(result).toEqual({ nodes: path.map((node) => node.id) });
    });
  });

  describe('getNearestNode', () => {
    it('should return nearest node as ID when returnType is SHALLOW', async () => {
      graphService.getNearestNode.mockResolvedValue('node123');

      const result = await navigationService.getNearestNode(
        1,
        2,
        3,
        0.5,
        NearestReturnType.SHALLOW,
      );
      expect(graphService.getNearestNode).toHaveBeenCalledWith(
        1,
        2,
        3,
        0.5,
        NearestReturnType.SHALLOW,
      );
      expect(result).toBe('node123');
    });

    it('should return nearest node object when returnType is DEEP', async () => {
      const node: GraphNode = {
        id: 'node123',
        label: 'Nearest Node',
        x: 1,
        y: 2,
        z: 3,
        graphId: 'graph1',
        edges: [],
      };
      graphService.getNearestNode.mockResolvedValue(node);

      const result = await navigationService.getNearestNode(
        1,
        2,
        3,
        undefined,
        NearestReturnType.DEEP,
      );
      expect(graphService.getNearestNode).toHaveBeenCalledWith(
        1,
        2,
        3,
        undefined,
        NearestReturnType.DEEP,
      );
      expect(result).toBe(node);
    });

    it('should return null if no nearest node found', async () => {
      graphService.getNearestNode.mockResolvedValue(null);

      const result = await navigationService.getNearestNode(1, 2, 3);
      expect(result).toBeNull();
    });
  });
});
